<?php
require_once __DIR__ . '/../init.php';

// 🛑 التحقق من صلاحيات المدير العام أو المدير المقيد
if (!isset($_SESSION['user']) || ($_SESSION['user']['role'] !== 'super_admin' && $_SESSION['user']['role'] !== 'admin')) {
    header('Location: /barcode-system/admin/index.php');
    exit;
}

// إعدادات الصفحات
$limit = 100; // عدد الباركودات في الصفحة الواحدة
$page = isset($_GET['p']) ? (int)$_GET['p'] : 1;
if ($page < 1) $page = 1;
$offset = ($page - 1) * $limit;

// جلب الباركودات من قاعدة البيانات
$stmt = $pdo->prepare("SELECT barcode FROM products ORDER BY barcode ASC LIMIT :limit OFFSET :offset");
$stmt->bindParam(':limit', $limit, PDO::PARAM_INT);
$stmt->bindParam(':offset', $offset, PDO::PARAM_INT);
$stmt->execute();
$barcodes = $stmt->fetchAll(PDO::FETCH_COLUMN);

// جلب العدد الإجمالي للصفحات
$total_stmt = $pdo->query("SELECT COUNT(*) FROM products");
$total_rows = $total_stmt->fetchColumn();
$total_pages = ceil($total_rows / $limit);
?>
<!doctype html>
<html lang="ar">
<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width,initial-scale=1">
    <title>طباعة الباركودات</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <script src="https://cdn.jsdelivr.net/npm/jsbarcode@3.11.6/dist/JsBarcode.all.min.js"></script>
    <style>
    body { font-family: Tahoma, Arial, sans-serif; direction: rtl; padding: 20px; }
    
    .barcode-container {
        display: inline-block;
        margin: 20px 15px;
        text-align: center;
        border: 1px solid #ddd;
        padding: 10px;
        width: 150px; /* تم تقليل العرض مرة أخرى لضمان عدم الخروج من الصفحة المطبوعة */
        max-width: 100%; /* ضمان التجاوب على الجوال */
        border-radius: 6px;
        box-shadow: 0 3px 8px rgba(0,0,0,0.05);
    }

    /* 🛑 الحل: اجعل وسم SVG يحتل 100% من عرض الحاوية */
    .barcode-container svg {
        width: 100%;
        height: auto;
        display: block;
        margin: 0;
        padding: 0;
    }
    
    .barcode-text {
        font-size: 11px;
        margin-top: 8px;
        font-weight: bold;
        color: #333;
        word-break: break-all;
    }
    
    /* تنسيقات خاصة بالطباعة */
    @media print {
        body { 
            padding: 0; 
            margin: 0;
        }
        .no-print {
            display: none !important;
        }
        .barcode-container {
            page-break-inside: avoid;
            margin: 10px 8px;
            box-shadow: none;
            border: 1px solid #000;
        }
    }
</style>
</head>
<body>
    <div class="container-fluid">
        <div class="no-print mb-4">
            <h4>طباعة الباركودات (الصفحة: <?= $page ?> من <?= $total_pages ?>)</h4>
            <a href="/barcode-system/admin_dashboard.php" class="btn btn-secondary me-2">العودة للوحة الإدارة</a>
            <button onclick="window.print()" class="btn btn-primary me-2">🖨️ طباعة هذه الصفحة</button>
            <p class="mt-2 text-danger">ملاحظة: يمكنك طباعة 100 باركود في كل صفحة.</p>
        </div>

        <div>
    <?php foreach ($barcodes as $barcode): ?>
        <div class="barcode-container">
            <svg id="barcode-<?= htmlspecialchars($barcode) ?>" class="barcode-code"></svg>
            <div class="barcode-text"><?= htmlspecialchars($barcode) ?></div>
        </div>
    <?php endforeach; ?>
</div>

        <nav class="no-print mt-4">
            <ul class="pagination justify-content-center">
                <?php if ($page > 1): ?>
                    <li class="page-item"><a class="page-link" href="?p=<?= $page - 1 ?>">السابق</a></li>
                <?php endif; ?>
                
                <?php 
                // عرض 5 صفحات حول الصفحة الحالية
                $start = max(1, $page - 2);
                $end = min($total_pages, $page + 2);

                for ($i = $start; $i <= $end; $i++): 
                ?>
                    <li class="page-item <?= ($i == $page) ? 'active' : '' ?>">
                        <a class="page-link" href="?p=<?= $i ?>"><?= $i ?></a>
                    </li>
                <?php endfor; ?>
                
                <?php if ($page < $total_pages): ?>
                    <li class="page-item"><a class="page-link" href="?p=<?= $page + 1 ?>">التالي</a></li>
                <?php endif; ?>
            </ul>
        </nav>

    </div>

   <script>
    document.addEventListener("DOMContentLoaded", function() {
        <?php foreach ($barcodes as $barcode): ?>
            JsBarcode("#barcode-<?= htmlspecialchars($barcode) ?>", "<?= htmlspecialchars($barcode) ?>", {
                format: "CODE128", 
                displayValue: false,
                margin: 0, 
                /* 🛑 الحل: تقليل عرض خطوط الباركود وترك الحاوية تتحكم بالعرض الكلي */
                width: 1, // قيمة نحيفة جداً (تساعد على التناسب)
                height: 50 // ارتفاع جيد للباركود
            });
        <?php endforeach; ?>
    });
</script>
</body>
</html>