<?php
// FILE: admin/products_list.php - الكود الكامل بعد جعل المقاس مطلوبًا
require_once __DIR__ . '/../init.php';

// 1. التحقق من صلاحيات الدخول: مسموح لـ super_admin و admin
if (!isset($_SESSION['user']) || ($_SESSION['user']['role'] !== 'super_admin' && $_SESSION['user']['role'] !== 'admin')) {
    header('Location: /barcode-system/admin/login.php');
    exit;
}

// 🛑 تحديد الدور وتطبيق القيد والقيمة الافتراضية
$user_role = $_SESSION['user']['role'] ?? 'admin'; 

// إذا كان المستخدم هو 'admin' (وليس super_admin)، اجعل الحقل للقراءة فقط
$production_date_readonly = ($user_role === 'admin') ? 'readonly' : ''; 

// القيمة الافتراضية لتاريخ الإنتاج هي تاريخ اليوم
$default_production_date = date('Y-m-d'); 


// 🛑 إعدادات الترقيم (Pagination) 🛑
$products_per_page = 10;
$current_page = max(1, intval($_GET['page'] ?? 1)); // تأكد من أن الصفحة لا تقل عن 1

// 🎯 (أ) جلب العدد الكلي للمنتجات
$total_products_stmt = $pdo->query("SELECT COUNT(*) FROM products");
$total_products = $total_products_stmt->fetchColumn();

// 🎯 (ب) حساب عدد الصفحات الكلي
$total_pages = ceil($total_products / $products_per_page);

// 🎯 (ت) حساب نقطة البداية (OFFSET)
$start_from = ($current_page - 1) * $products_per_page;
if($start_from < 0) $start_from = 0; // تأمين ضد القيم السالبة

// 🎯 (ث) استعلام جلب المنتجات المحدود (مع الترتيب والتحديد)
$rows = $pdo->query("
    SELECT *, delivery_note_ref, delivery_date 
    FROM products 
    ORDER BY created_at DESC 
    LIMIT $start_from, $products_per_page
")->fetchAll(PDO::FETCH_ASSOC);

// --------------------------------------------------------------------
// 🆕 جلب القوائم المنسدلة (لنموذج الإضافة)
// --------------------------------------------------------------------

// 1. جلب جميع أسماء المنتجات الفريدة
$stmt_names = $pdo->query('SELECT DISTINCT name FROM products WHERE name IS NOT NULL AND name != "" ORDER BY name ASC');
$product_names = $stmt_names->fetchAll(PDO::FETCH_COLUMN);

// 2. جلب جميع المقاسات الفريدة
$stmt_sizes = $pdo->query('SELECT DISTINCT size FROM products WHERE size IS NOT NULL AND size != "" ORDER BY size ASC');
$product_sizes = $stmt_sizes->fetchAll(PDO::FETCH_COLUMN);

// --------------------------------------------------------------------

// التعامل مع إضافة المنتج عبر POST
$error = $success = '';
if($_SERVER['REQUEST_METHOD']==='POST'){
    $barcode = trim($_POST['barcode'] ?? '');
    $name = trim($_POST['name'] ?? '');
    $size = trim($_POST['size'] ?? ''); // تم جعل هذا الحقل إجباريًا في HTML
    $production_date = $_POST['production_date'] ?: null;
    $notes = trim($_POST['notes'] ?? '');

    // التحقق، إضافة شرط المقاس
    if($_SESSION['user']['role'] !== 'super_admin' && $_SESSION['user']['role'] !== 'admin'){
        $error = "غير مصرح لك بإضافة منتجات.";
    } elseif(!$barcode || !$name || !$size){ // 🛑 شرط جديد للمقاس
        $error="الباركود واسم المنتج والمقاس مطلوبة.";
    } else {
        // التأكد من عدم وجود الباركود مسبقاً
        $check = $pdo->prepare("SELECT id FROM products WHERE barcode=?");
        $check->execute([$barcode]);
        if($check->fetch()){
            $error="رمز الباركود مسجل مسبقاً.";
        } else {
            try {
                $stmt = $pdo->prepare("INSERT INTO products (barcode, name, size, production_date, notes, created_by) VALUES (?,?,?,?,?,?)");
                $stmt->execute([$barcode, $name, $size, $production_date, $notes, $_SESSION['user']['id']]); // تم حذف ?: null من size
                $success = "تمت إضافة المنتج بنجاح. معرف: " . $pdo->lastInsertId();
                // إعادة توجيه لمنع إرسال النموذج مرة أخرى عند تحديث الصفحة
                header("Location: " . $_SERVER['PHP_SELF'] . "?success=" . urlencode($success));
                exit;
            } catch (Exception $e) {
                $error = "حدث خطأ أثناء الحفظ: " . $e->getMessage();
            }
        }
    }
}

// عرض رسائل النجاح أو الخطأ بعد إعادة التوجيه
if(isset($_GET['success'])){
    $success = htmlspecialchars($_GET['success']);
}
?>
<!doctype html>
<html lang="ar">
<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width,initial-scale=1">
    <title>لوحة الإدارة - Backcare</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.min.css">
<style>
    body{font-family:Tajawal,sans-serif;background:#7fb8ffa;padding:16px;direction:rtl;}
    .card{border-radius:12px;padding:16px;margin-bottom:16px;box-shadow:0 4px 12px rgba(0,0,0,0.05);}
    .table-responsive{overflow-x:auto;}
    
    /* 🟢 تنسيقات الألوان الجديدة لحقول الإدخال */
    .form-control, .form-select { 
        background-color: #d9eaff !important; 
        border: 1px solid #d1d5db; 
        transition: border-color 0.2s;
    }
    .form-control:focus, .form-select:focus {
        border-color: #a0c3ff; 
        box-shadow: 0 0 0 0.25rem rgba(0, 123, 255, 0.1);
    }
    
    /* 🟢 تنسيق الـ input-group لـ RTL لظهور الأزرار على اليسار */
    /* التنسيق الأساسي للحقول المصاحبة بزر واحد */
    .input-group .form-control, .input-group .form-select { 
        /* الزاوية اليمنى (البداية) مدورة */
        border-top-right-radius: 0.375rem !important; 
        border-bottom-right-radius: 0.375rem !important;
        /* الزاوية اليسرى (النهاية) غير مدورة */
        border-top-left-radius: 0 !important;
        border-bottom-left-radius: 0 !important;
        text-align: right; 
    }
    /* الزر المصاحب (في أقصى اليسار) يجب أن تكون زاويته مدورة */
    .input-group .btn-submit-box { 
        border-top-left-radius: 0.375rem !important; 
        border-bottom-left-radius: 0.375rem !important;
    }

    /* 🟢 تنسيق حقل الباركود (يحتوي على زرين) */
    .input-group #barcodeInput {
        /* الحقل يحتوي على زرين، لذا زواياه اليسرى يجب أن لا تكون مدورة */
        border-top-left-radius: 0 !important;
        border-bottom-left-radius: 0 !important;
    }
    /* زر السهم المعقوف (الآن في المنتصف) يجب أن يكون بدون زوايا */
    .input-group .btn-barcode-submit {
          border-radius: 0 !important; 
          background-color: #4a7dce !important; 
          border-color: #4a7dce !important;
    }
    /* زر الكاميرا (الآن في أقصى اليسار) يجب أن تكون زاويته مدورة */
    .input-group #scanBtn { 
        border-top-left-radius: 0.375rem !important; 
        border-bottom-left-radius: 0.375rem !important;
        background-color: #209353 !important;
        border-color: #209353 !important;
    }
    /* تعديل الزرار المصاحب لبقية الحقول ليتناسب مع درجة لون السهم */
    .input-group .btn-submit-box {
        background-color: #4a7dce !important; 
        border-color: #4a7dce !important;
    }
    
</style>
</head>
<body>
    <div class="container">
        
        <div class="mb-3">
    <h2 class="h4 mb-3">لوحة الإدارة</h2>
    <div class="d-grid gap-2 d-md-flex justify-content-md-start mb-3">
        
        <?php if ($_SESSION['user']['role'] === 'super_admin'): ?> 
            <a href="/barcode-system/admin/manage_delivery_notes.php" class="btn btn-primary">📋 إدارة السندات</a>
            
            <a href="/barcode-system/admin/generate_barcode_images.php" class="btn btn-info text-white">🖨️ طباعة المسجل</a>
            
            <a href="/barcode-system/admin/generate_temporary_barcodes.php" class="btn btn-success">📄 طباعة (100K+)</a>
        
            <a href="/barcode-system/admin/manage_users.php" class="btn btn-success">➕ إدارة المستخدمين</a>
        
            <a href="/barcode-system/admin/export_csv.php" class="btn btn-secondary">📥 تصدير CSV</a>
        <?php endif; ?>

        <a href="/barcode-system/admin/change_password.php" class="btn btn-warning">تغيير كلمة المرور</a>
        <a href="/barcode-system/auth.php?action=logout" class="btn btn-danger">تسجيل خروج</a>
    </div>
</div>

<?php if ($error): ?>
    <div class="alert alert-danger"><?= $error ?></div>
<?php endif; ?>
<?php if ($success): ?>
    <div class="alert alert-success"><?= $success ?></div>
<?php endif; ?>

<div class="card">
    <div class="d-flex justify-content-end mb-3">
        <a href="/barcode-system/admin_dashboard.php" class="btn btn-primary btn-sm">
            <i class="bi bi-house-door-fill me-2"></i> العودة للوحة التحكم
        </a>
    </div>
    
    <h4>إضافة منتج جديد</h4>
    <?php if ($_SESSION['user']['role'] === 'super_admin' || $_SESSION['user']['role'] === 'admin'): ?>
    <form method="post" id="addProductForm">
        <div class="row">
            
            <div class="col-12 col-md-6 mb-2">
                <label class="form-label">باركود المنتج (مطلوب)</label>
                <div class="input-group">
                    
                    <input name="barcode" id="barcodeInput" class="form-control" placeholder="أدخل الباركود يدوياً" required value="<?= htmlspecialchars($_POST['barcode'] ?? '') ?>">
                    
                    <button type="button" class="btn btn-primary btn-barcode-submit" title="رمز تأكيد الباركود">
                        <i class="bi bi-arrow-return-left"></i>
                    </button>
                    
                    <button type="button" id="scanBtn" class="btn btn-success" title="مسح الباركود بالكاميرا">
                        <i class="bi bi-camera-fill"></i>
                    </button>
                    
                </div>
                
                <video id="video" width="100%" height="auto" autoplay style="border-radius:8px; display:none; max-width: 320px;"></video>
            </div>
            
            <div class="col-md-6 mb-2">
                <label class="form-label" for="productNameInput">اسم المنتج (مطلوب)</label>
                <div class="input-group">
                    <input name="name" 
                           id="productNameInput" 
                           class="form-control" 
                           placeholder="اكتب أو اختر اسم المنتج" 
                           list="nameSuggestions" 
                           required 
                           value="<?= htmlspecialchars($_POST['name'] ?? '') ?>">
                    
                    <datalist id="nameSuggestions">
                        <?php foreach ($product_names as $name): ?>
                            <option value="<?= htmlspecialchars($name) ?>">
                        <?php endforeach; ?>
                    </datalist>
                    
                    <button type="button" class="btn btn-primary btn-submit-box" title="رمز">
                        <i class="bi bi-arrow-return-left"></i>
                    </button>
                </div>
            </div>

            
            <div class="col-md-6 mb-2">
                <label class="form-label" for="productSizeInput">المقاس (Size) (مطلوب)</label>
                <div class="input-group">
                    <input name="size" 
                           id="productSizeInput" 
                           class="form-control" 
                           placeholder="اكتب أو اختر المقاس" 
                           list="sizeSuggestions" 
                           required 
                           value="<?= htmlspecialchars($_POST['size'] ?? '') ?>">
                    
                    <datalist id="sizeSuggestions">
                        <?php foreach ($product_sizes as $size): ?>
                            <option value="<?= htmlspecialchars($size) ?>">
                        <?php endforeach; ?>
                    </datalist>
                    
                    <button type="button" class="btn btn-primary btn-submit-box" title="رمز">
                        <i class="bi bi-arrow-return-left"></i>
                    </button>
                </div>
            </div>
            
            
            <div class="col-md-6 mb-2">
                <label class="form-label">تاريخ الإنتاج</label>
                <div class="input-group">
                    <input name="production_date" 
                            type="date" 
                            class="form-control" 
                            value="<?= htmlspecialchars($_POST['production_date'] ?? $default_production_date) ?>"
                            <?= $production_date_readonly ?>>
                    <button type="button" class="btn btn-primary btn-submit-box" title="رمز">
                        <i class="bi bi-arrow-return-left"></i>
                    </button>
                </div>
            </div>
            
            <div class="col-md-6 mb-2">
                <label class="form-label">ملاحظات</label>
                <div class="input-group">
                    <textarea name="notes" class="form-control"><?= htmlspecialchars($_POST['notes'] ?? '') ?></textarea>
                    <button type="button" class="btn btn-primary btn-submit-box" title="رمز">
                        <i class="bi bi-arrow-return-left"></i>
                    </button>
                </div>
            </div>
        </div>
        
        <button type="submit" class="btn btn-primary mt-2">حفظ المنتج</button>
    </form>
    <?php else: ?>
        <div class="alert alert-info">صلاحيتك تسمح بالعرض فقط، لا يمكنك إضافة منتجات.</div>
    <?php endif; ?>
</div>
    
<?php 
// تعريف الألوان الفاتحة للأعمدة
$col_colors = [
    'id' => '#f8f9fa',
    'barcode' => '#e6f0ff',
    'name' => '#ffffff',
    'size' => '#e6fffd',
    'production_date' => '#fffbe6',
    'expiry_date' => '#ffebeb', 
    'notes' => '#f5f5f5',
    'created_by' => '#e6f7ff',
    'created_at' => '#e6e6ff',
    'delivery_note_ref' => '#d0e0fb',
    'delivery_date' => '#e6ffef',
];
?>

<div class="card">
    <h4 class="mb-3">قائمة المنتجات (<span class="text-primary"><?= $total_products ?></span>) - الصفحة الحالية: <?= $current_page ?></h4>
    <div class="table-responsive">
        <table class="table table-hover table-sm">
            
            <thead style="font-size: 0.9rem;">
                <tr style="border-bottom: 3px solid #ccc;">
                    <th scope="col" class="text-white text-center" style="width: 50px; background-color: #6c757d !important;">ID</th>
                    <th scope="col" class="text-white text-center" style="background-color: #0d6efd !important;">الباركود</th>
                    <th scope="col" class="text-white" style="background-color: #198754 !important;">الاسم</th>
                    <th scope="col" class="text-white text-center" style="width: 80px; background-color: #20c997 !important;">المقاس</th>
                    <th scope="col" class="text-center" style="background-color: #ffc107 !important; color: #333 !important;">تاريخ انتاج</th>
                    <th scope="col" class="text-white text-center" style="background-color: #007bff !important;">رقم السند</th>
                    <th scope="col" class="text-white text-center" style="background-color: #28a745 !important;">تاريخ التسليم</th>
                    <th scope="col" class="text-white" style="background-color: #6f42c1 !important;">الملاحظات</th>
                    <th scope="col" class="text-center" style="background-color: #0dcaf0 !important; color: #333 !important;">بواسطة</th>
                    <th scope="col" class="text-white text-center" style="background-color: #000080 !important;">تاريخ الإنشاء</th>
                    <th scope="col" class="text-white text-center" style="width: 120px; background-color: #343a40 !important;">إجراءات</th>
                </tr>
            </thead>
            
            <tbody>
                <?php 
                $user_list = $pdo->query("SELECT id, username FROM users")->fetchAll(PDO::FETCH_KEY_PAIR);
                
                foreach($rows as $r): 
                    $row_bg_color = ''; 
                ?>
                <tr style="<?= $row_bg_color ?>">
                    <td class="small text-center" style="background-color: <?= $col_colors['id'] ?> !important;"><?= $r['id'] ?></td>
                    <td class="small text-center text-break" style="background-color: <?= $col_colors['barcode'] ?> !important;"><strong><?= htmlspecialchars($r['barcode']) ?></strong></td>
                    <td class="small" style="background-color: <?= $col_colors['name'] ?> !important;"><?= htmlspecialchars($r['name']) ?></td>
                    <td class="small text-center" style="background-color: <?= $col_colors['size'] ?> !important;"><?= htmlspecialchars($r['size'] ?? '-') ?></td>
                    <td class="small text-center" style="background-color: <?= $col_colors['production_date'] ?> !important;"><?= htmlspecialchars($r['production_date'] ?? '-') ?></td>
                    
                    <td class="small text-center fw-bold" style="background-color: <?= $col_colors['delivery_note_ref'] ?> !important;">
                        <?= htmlspecialchars($r['delivery_note_ref'] ?? '-') ?>
                    </td>
                    <td class="small text-center" style="background-color: <?= $col_colors['delivery_date'] ?> !important;">
                        <?= htmlspecialchars($r['delivery_date'] ?? '-') ?>
                    </td>
                    
                    <td class="small" style="background-color: <?= $col_colors['notes'] ?> !important;">
                        <div class="text-truncate" style="max-width: 150px;" title="<?= htmlspecialchars($r['notes'] ?: 'لا توجد ملاحظات') ?>">
                            <?= htmlspecialchars($r['notes'] ?? '-') ?>
                        </div>
                    </td>
                    
                    <td class="small text-center text-muted" style="background-color: <?= $col_colors['created_by'] ?> !important;"><?= htmlspecialchars($user_list[$r['created_by']] ?? 'N/A') ?></td>
                    <td class="small text-center text-muted" style="background-color: <?= $col_colors['created_at'] ?> !important;"><?= date('Y-m-d', strtotime($r['created_at'])) ?></td>
                    
                    <td class="text-center" style="background-color: #f8f9fa !important;">
                        <?php if ($_SESSION['user']['role'] === 'super_admin'): ?>
                            <a href="edit_product.php?id=<?= $r['id'] ?>" class="btn btn-sm btn-info me-1 py-0">تعديل</a>
                            <button onclick="deleteProduct(<?= $r['id'] ?>)" class="btn btn-sm btn-danger py-0">حذف</button>
                        <?php else: ?>
                            <a href="edit_product.php?id=<?= $r['id'] ?>" class="btn btn-sm btn-info me-1 py-0">عرض</a>
                        <?php endif; ?>
                    </td>
                </tr>
                <?php endforeach;?>
            </tbody>
        </table>
    </div>
    
    <?php if ($total_pages > 1): ?>
    <nav aria-label="Products Pagination">
        <ul class="pagination justify-content-center mt-4 flex-wrap">
            
            <li class="page-item <?= ($current_page <= 1) ? 'disabled' : '' ?>">
                <a class="page-link" href="?page=1">الصفحة الأولى</a>
            </li>

            <li class="page-item <?= ($current_page <= 1) ? 'disabled' : '' ?>">
                <a class="page-link" href="?page=<?= max(1, $current_page - 1) ?>" aria-label="Previous">
                    <span aria-hidden="true">&laquo;</span>
                </a>
            </li>
            
            <?php
            // تحديد نطاق الأرقام المعروضة (مثلاً، 5 صفحات حول الصفحة الحالية)
            $range = 2; // عرض صفحتين قبل وبعد
            $start_loop = max(1, $current_page - $range);
            $end_loop = min($total_pages, $current_page + $range);
            
            if ($start_loop > 1) { echo '<li class="page-item disabled"><span class="page-link">...</span></li>'; }
            
            for ($i = $start_loop; $i <= $end_loop; $i++): 
            ?>
                <li class="page-item <?= ($i == $current_page) ? 'active' : '' ?>">
                    <a class="page-link" href="?page=<?= $i ?>"><?= $i ?></a>
                </li>
            <?php endfor;
            
            if ($end_loop < $total_pages) { echo '<li class="page-item disabled"><span class="page-link">...</span></li>'; }
            ?>
            
            <li class="page-item <?= ($current_page >= $total_pages) ? 'disabled' : '' ?>">
                <a class="page-link" href="?page=<?= min($total_pages, $current_page + 1) ?>" aria-label="Next">
                    <span aria-hidden="true">&raquo;</span>
                </a>
            </li>
            
            <li class="page-item <?= ($current_page >= $total_pages) ? 'disabled' : '' ?>">
                <a class="page-link" href="?page=<?= $total_pages ?>">الصفحة الأخيرة</a>
            </li>
        </ul>
    </nav>
    <?php endif; ?>

</div>
    
    <script src="https://unpkg.com/@zxing/library@0.18.6/umd/index.min.js"></script>
    <script>
        // 1. كود الحذف (deleteProduct)
        async function deleteProduct(id) {
            // ... (بقية كود الحذف)
        }
        
        // --------------------------------------------------------------------
        // 🟢 منطق الانتقال بين الحقول (مع التحقق من القيمة) 🟢
        // --------------------------------------------------------------------
        
        const form = document.getElementById('addProductForm');
        // جلب جميع حقول الإدخال ذات الصلة بترتيبها في النموذج
        const inputs = [
            document.getElementById('barcodeInput'),
            document.getElementById('productNameInput'),
            document.getElementById('productSizeInput'),
            form.querySelector('input[name="production_date"]'),
            form.querySelector('textarea[name="notes"]')
        ].filter(el => el); // فلترة العناصر للتأكد من وجودها
        
        // جلب جميع الأزرار الصغيرة التي تحمل فئة btn-submit-box أو btn-barcode-submit
        const submitButtons = form.querySelectorAll('.btn-submit-box, .btn-barcode-submit');

        submitButtons.forEach(button => {
            // تأكد من أن الزر من نوع button وليس زر الكاميرا (scanBtn)
            if (button.id !== 'scanBtn') {
                button.addEventListener('click', (e) => {
                    e.preventDefault(); // منع أي سلوك افتراضي
                    
                    const currentInputGroup = button.closest('.input-group');
                    if (!currentInputGroup) return;

                    const currentInput = currentInputGroup.querySelector('input, textarea, select');
                    if (!currentInput) return;
                    
                    // 🎯 التحقق من صلاحية الحقل الحالي 
                    // نستخدم checkValidity() المدمجة التي تتحقق من سمة 'required'
                    if (currentInput.checkValidity() === false) {
                        // إذا كان الحقل غير صالح (فارغ وكان مطلوباً)، نعرض رسالة الخطأ المدمجة في المتصفح
                        currentInput.reportValidity();
                        currentInput.focus();
                        return; // 🛑 التوقف هنا وعدم الانتقال
                    }

                    // إذا كان الحقل صالحاً، نستمر في عملية الانتقال
                    const currentIndex = inputs.findIndex(input => input === currentInput);
                    
                    if (currentIndex !== -1 && currentIndex < inputs.length - 1) {
                        const nextInput = inputs[currentIndex + 1];
                        if (nextInput) {
                            nextInput.focus();
                            if (nextInput.tagName === 'INPUT' || nextInput.tagName === 'TEXTAREA') {
                                nextInput.select(); 
                            }
                        }
                    } else if (currentIndex === inputs.length - 1) {
                        // إذا كان الحقل الحالي هو الأخير، نركز على زر الحفظ الرئيسي
                        form.querySelector('button[type="submit"]').focus();
                    }
                });
            }
        });

        // --------------------------------------------------------------------
        
        // 2. كود المسح الضوئي 
        const codeReader = new ZXing.BrowserBarcodeReader();
        const video = document.getElementById('video');
        const scanBtn = document.getElementById('scanBtn');
        const barcodeInput = document.getElementById('barcodeInput');
        const scanBtnIcon = scanBtn.querySelector('i'); 
        let active = false;

        scanBtn.addEventListener('click', () => {
            if (active) {
                // ... (إيقاف الكاميرا)
                codeReader.reset();
                video.style.display = 'none';
                scanBtnIcon.className = 'bi bi-camera-fill';
                active = false;
                return;
            }

            // بدء الكاميرا
            video.style.display = 'block';
            scanBtnIcon.className = 'bi bi-camera-video-fill';
            active = true;

            // استخدام decodeFromConstraints
            codeReader.decodeFromConstraints({
                video: {
                    width: { ideal: 640 }, 
                    height: { ideal: 480 },
                    facingMode: "environment" 
                }
            }, video, (result, err) => {
                if (result) {
                    barcodeInput.value = result.text.trim();
                    
                    // إيقاف الكاميرا فوراً بعد القراءة
                    codeReader.reset();
                    video.style.display = 'none';
                    scanBtnIcon.className = 'bi bi-camera-fill';
                    active = false;
                    
                    // عند المسح الناجح: ننقل التركيز إلى حقل اسم المنتج
                    const nameInput = document.getElementById('productNameInput');
                    if (nameInput) {
                        nameInput.focus();
                        nameInput.select();
                    } else {
                        document.getElementById('addProductForm').submit();
                    }
                }

                if (err && !(err instanceof ZXing.NotFoundException)) {
                    // ...
                }
            });
        });

    </script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>