<?php
// FILE: admin/qr_generator/generate.php - النسخة النهائية (إصلاح الكاش + التسمية الإنجليزية)

// 1. 🟢 إضافة تحميل مكتبات Composer
require_once __DIR__ . '/../../vendor/autoload.php';

// 2. تضمين ملف إعدادات النظام الأساسي (للوصول إلى $pdo)
require_once __DIR__ . '/../../init.php';

// التأكد من أن المستخدم مسجل الدخول قبل توليد الرموز
if (!isset($_SESSION['user'])) {
    http_response_code(403);
    exit("Forbidden");
}

// استخدام مكتبة Endroid QrCode
use Endroid\QrCode\QrCode;
use Endroid\QrCode\Writer\PngWriter;
use Endroid\QrCode\Color\Color;
use Endroid\QrCode\Encoding\Encoding;
use Endroid\QrCode\Label\Label;
use Endroid\QrCode\Logo\Logo;
use Endroid\QrCode\ErrorCorrectionLevel; 
use Endroid\QrCode\Exception\FileNotFoundException; 

// **********************************************
// 1. استقبال البيانات وتصفيتها
// **********************************************
$url = filter_var($_GET['url'] ?? '', FILTER_SANITIZE_URL);
// إزالة القيمة الافتراضية، والحصول على التسمية كقيمة نصية
$productName = trim($_GET['productName'] ?? ''); 

// 🔴 التحقق الإجباري من وجود التسمية (productName)
if (empty($productName)) {
    header('Content-Type: text/plain');
    http_response_code(400); // Bad Request
    echo "QR Generation Error: يجب تزويد اسم المنتج (productName) لتوليد رمز QR.";
    exit;
}

// توليد قيمة الهاش (Checksum) بناءً على الـ URL
$checksum = md5($url);

try {
    // نستخدم متغير الاتصال $pdo المأخوذ من init.php
    
    // **********************************************
    // 2. التحقق من التخزين المؤقت (Check Cache)
    // **********************************************
    // جلب بيانات الصورة والتسمية
    $stmt = $pdo->prepare("SELECT image_data, product_name FROM qr_codes WHERE checksum = ?");
    $stmt->execute([$checksum]);
    $cachedData = $stmt->fetch(PDO::FETCH_ASSOC); 

    // 🔴 التعديل الحاسم: نتحقق من وجود السجل *و* أن بيانات الصورة ليست فارغة (NULL)
    if ($cachedData && !empty($cachedData['image_data'])) {
        // إذا وُجدت بيانات الصورة في قاعدة البيانات، أرسلها مباشرة
        header('Content-Type: image/png');
        echo $cachedData['image_data'];
        exit;
    }

    // **********************************************
    // 3. التوليد (Generate)
    // **********************************************
    
    // إعداد رمز QR
    $qrCode = QrCode::create($url)
        ->setEncoding(new Encoding('UTF-8'))
        ->setErrorCorrectionLevel(ErrorCorrectionLevel::High)
        // ***** حجم عالي لجودة الطباعة *****
        ->setSize(1000) 
        ->setMargin(10)
        ->setForegroundColor(new Color(50, 50, 50))
        ->setBackgroundColor(new Color(255, 255, 255));

    // إعداد الشعار (المسار بالنسبة للجذر)
    $logoFilePath = __DIR__ . '/../../assets/backcare-logo.png'; 
    $logo = null; 

    try {
        if (file_exists($logoFilePath)) {
            $logo = Logo::create($logoFilePath)
                ->setResizeToWidth(250) 
                ->setPunchoutBackground(true);
        } 
    } catch (FileNotFoundException $e) {
        // تجاهل
    } catch (\Throwable $e) {
        // تجاهل
    }

    // 🟢 تفعيل التسمية (Label) بالإنجليزية
    $label = null; 
    
    // تأكد من وجود اسم المنتج
    if (!empty($productName)) {
        // محاولة استخدام خط موجود
        $fontFilePath = __DIR__ . '/../../fonts/Arial.ttf';
        
        // التحقق من وجود الخط، وإلا استخدام الاسم الافتراضي
        $font = file_exists($fontFilePath) ? $fontFilePath : 'Arial';
        
        $label = Label::create($productName)
            ->setTextColor(new Color(0, 0, 0))
            ->setFont(new \Endroid\QrCode\Label\Font\Font($font, 40)); 
    }

    // توليد الصورة والحصول على البيانات الثنائية
    $writer = new PngWriter();
    $result = $writer->write($qrCode, $logo, $label);
    $imageData = $result->getString();

    // **********************************************
    // 4. التخزين في قاعدة البيانات (Store in Cache)
    // **********************************************
    
    // 🔴 منطق التخزين المحدث: إذا كان السجل موجوداً (أي سجل التسمية موجود)، نقوم بالتحديث بدلاً من الإدخال
    if ($cachedData) {
        // السجل موجود بالفعل لكن بدون صورة (بعد التعديل في صفحة الإدارة)، نقوم بتحديث بيانات الصورة
        $stmt = $pdo->prepare("UPDATE qr_codes SET image_data = ?, product_name = ? WHERE checksum = ?");
        $stmt->bindValue(1, $imageData, PDO::PARAM_LOB);
        $stmt->bindValue(2, $productName);
        $stmt->bindValue(3, $checksum);
        $stmt->execute();
    } else {
        // السجل غير موجود، نقوم بإدخال جديد
        $stmt = $pdo->prepare("INSERT INTO qr_codes (url_target, image_data, checksum, product_name) VALUES (?, ?, ?, ?)");
        $stmt->bindValue(1, $url);
        $stmt->bindValue(2, $imageData, PDO::PARAM_LOB);
        $stmt->bindValue(3, $checksum);
        $stmt->bindValue(4, $productName);
        $stmt->execute();
    }
    
    // **********************************************
    // 5. إرسال الصورة
    // **********************************************
    header('Content-Type: ' . $result->getMimeType());
    echo $imageData;

} catch (PDOException $e) {
    // في حالة فشل الاتصال بقاعدة البيانات
    header('Content-Type: text/plain');
    http_response_code(500);
    echo "Database Error: " . $e->getMessage();
    exit;
} catch (Exception $e) {
     // في حالة أي خطأ آخر (مثل عدم وجود GD)
    header('Content-Type: text/plain');
    http_response_code(500);
    echo "QR Generation Error: " . $e->getMessage();
    exit;
}
?>