<?php
// FILE: admin/manage_users.php
require_once __DIR__ . '/../init.php';

// 🛑 التحقق من صلاحيات المدير العام فقط (Super Admin)
if (!isset($_SESSION['user']) || $_SESSION['user']['role'] !== 'super_admin') {
    // تصحيح التوجيه إلى لوحة التحكم الجديدة
    header('Location: /barcode-system/admin_dashboard.php');
    exit;
}

// --------------------------------------------------------------------
// 1. تعريف الصلاحيات المتاحة (المحدثة لتشمل كل روابط اللوحة)
// --------------------------------------------------------------------
$all_permissions = [
    'products' => [
        'label' => 'إدارة المنتجات',
        'perms' => [
            'products_view'         => '1. عرض قائمة المنتجات',
            'products_add'          => '2. إضافة منتجات جديدة',
            'products_export'       => '3. تصدير قائمة المنتجات (CSV)',
            'products_delete'       => '4. حذف المنتجات',
        ]
    ],
    'barcode_qr' => [
        'label' => 'الباركود و QR Code',
        'perms' => [
            'barcode_generate_print'=> '1. توليد وطباعة الباركود',
            'barcode_heavy_print'   => '2. طباعة باركود (+100K)',
            'qr_manage'             => '3. إدارة (QR Code) المسجلة',
            'qr_generate_link'      => '4. مولد رمز QR للروابط',
        ]
    ],
    'delivery' => [
        'label' => 'إدارة التسليم والسندات',
        'perms' => [
            'delivery_note_manage'  => '1. إدارة سندات التسليم',
            'delivery_assign_product' => '2. تعيين منتج لسند تسليم',
        ]
    ],
    'user_management' => [
        'label' => 'إدارة المستخدمين',
        'perms' => [
            'users_view_manage'     => '1. عرض/إدارة المستخدمين',
        ]
    ],
    'reporting' => [
        'label' => 'التقارير والإعدادات',
        'perms' => [
            'reports_view'          => '1. عرض التقارير والإحصائيات',
        ]
    ]
];
// --------------------------------------------------------------------

$current_user_id = $_SESSION['user']['id'];
$message = isset($_GET['msg']) ? htmlspecialchars($_GET['msg']) : '';
$message_type = isset($_GET['type']) ? htmlspecialchars($_GET['type']) : '';
$edit_id = $_GET['edit_id'] ?? null; // ID المستخدم المراد تعديله

// 1. معالجة طلب حذف المستخدم
if (isset($_GET['action']) && $_GET['action'] === 'delete' && isset($_GET['id'])) {
    $delete_id = (int)$_GET['id'];

    if ($delete_id === $current_user_id) {
        $message = 'لا يمكنك حذف حسابك الحالي.';
        $message_type = 'danger';
    } else {
        try {
            $stmt = $pdo->prepare('DELETE FROM users WHERE id = ?');
            $stmt->execute([$delete_id]);

            if ($stmt->rowCount()) {
                $message = 'تم حذف المستخدم بنجاح.';
                $message_type = 'success';
            } else {
                $message = 'فشل حذف المستخدم أو لم يتم العثور عليه.';
                $message_type = 'warning';
            }
        } catch (\PDOException $e) {
            $message = 'خطأ في قاعدة البيانات: ' . $e->getMessage();
            $message_type = 'danger';
        }
    }
    // إعادة التوجيه لمنع إعادة الإرسال بعد الحذف
    header("Location: " . $_SERVER['PHP_SELF'] . "?msg=" . urlencode($message) . "&type=" . $message_type);
    exit;
}

// 2. معالجة طلب تحديث المستخدم (POST)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['user_id_to_edit'])) {
    $target_user_id = $_POST['user_id_to_edit'];
    $new_username = trim($_POST['new_username'] ?? '');
    $new_password = $_POST['new_password'] ?? '';
    $new_role = $_POST['new_role'] ?? '';
    // 🆕 جلب الصلاحيات المختارة من Checkboxes
    $selected_permissions = $_POST['permissions'] ?? [];

    // التحقق من أن السوبر أدمن لا يغير دوره إلى شيء آخر
    if ($target_user_id == $current_user_id && $new_role !== 'super_admin') {
        $message = 'لا يمكنك تغيير دورك كسوبر أدمن من هنا.';
        $message_type = 'danger';
    }
    // التحقق من أن حقل الاسم ليس فارغاً
    elseif (empty($new_username)) {
        $message = 'اسم المستخدم الجديد مطلوب.';
        $message_type = 'danger';
    }
    // التأكد من طول كلمة المرور الجديدة
    elseif (!empty($new_password) && strlen($new_password) < 6) {
        $message = 'يجب أن لا تقل كلمة المرور الجديدة عن 6 أحرف.';
        $message_type = 'danger';
    }
    else {
        $sql_parts = [];
        $params = [];

        $sql_parts[] = 'username = ?';
        $params[] = $new_username;

        $sql_parts[] = 'role = ?';
        $params[] = $new_role;

        // 🆕 تحويل مصفوفة الصلاحيات إلى JSON للحفظ
        if ($new_role === 'super_admin') {
            $permissions_json = null;
        } else {
        // 1. تجميع جميع الصلاحيات الصالحة في مصفوفة واحدة يدوياً
        $valid_perms = [];
        foreach ($all_permissions as $group) {
            // دمج مفاتيح الصلاحيات يدوياً لضمان أقصى توافق
            foreach (array_keys($group['perms']) as $perm_key) {
                $valid_perms[] = $perm_key;
            }
        }

        // 2. تصفية الصلاحيات باستخدام Anonymous Function لتوافق أوسع
        $final_permissions = array_filter($selected_permissions, function($p) use ($valid_perms) {
            return in_array($p, $valid_perms);
        });

        $permissions_json = json_encode($final_permissions);
    }

        $sql_parts[] = 'permissions_json = ?';
        $params[] = $permissions_json;


        // تحديث كلمة المرور (دون الحاجة للقديمة)
        if (!empty($new_password)) {
            $new_password_hash = password_hash($new_password, PASSWORD_DEFAULT);
            $sql_parts[] = 'password_hash = ?';
            $params[] = $new_password_hash;
        }

        $sql = 'UPDATE users SET ' . implode(', ', $sql_parts) . ' WHERE id = ?';
        $params[] = $target_user_id;

        try {
            $update_stmt = $pdo->prepare($sql);
            $update_stmt->execute($params);

            $message = 'تم تحديث بيانات المستخدم بنجاح. ' . (empty($new_password) ? '' : 'تم تغيير كلمة المرور أيضاً.');
            $message_type = 'success';

            // تحديث الجلسة إذا كان المستخدم هو السوبر أدمن نفسه
            if ($target_user_id == $current_user_id) {
                $_SESSION['user']['username'] = $new_username;
                $_SESSION['user']['role'] = $new_role;
                // 🆕 تحديث صلاحيات الجلسة
                if ($new_role !== 'super_admin' && $permissions_json !== null) {
                    $_SESSION['user']['permissions'] = json_decode($permissions_json, true) ?: [];
                } else {
                    // السوبر أدمن لديه صلاحيات كاملة افتراضياً
                    $_SESSION['user']['permissions'] = [];
                }
            }
        } catch (PDOException $e) {
            $message = 'خطأ في قاعدة البيانات: قد يكون اسم المستخدم مكرراً.';
            $message_type = 'danger';
        }
    }

    // إعادة التوجيه لمنع إعادة الإرسال بعد التحديث
    header("Location: " . $_SERVER['PHP_SELF'] . "?msg=" . urlencode($message) . "&type=" . $message_type);
    exit;
}

// 🗂️ جلب جميع المستخدمين
// 🆕 جلب عمود permissions_json أيضاً
$users_query = 'SELECT id, username, role, created_at, permissions_json FROM users ORDER BY id DESC';
$users = $pdo->query($users_query);
$user_list = $users->fetchAll(PDO::FETCH_ASSOC);

$edit_user = null;
$user_permissions = []; // 🆕 الصلاحيات الحالية للمستخدم المراد تعديله
if ($edit_id) {
    // 🆕 جلب عمود permissions_json أيضاً
    $edit_stmt = $pdo->prepare('SELECT id, username, role, permissions_json FROM users WHERE id = ?');
    $edit_stmt->execute([$edit_id]);
    $edit_user = $edit_stmt->fetch(PDO::FETCH_ASSOC);

    // 🆕 فك تشفير الصلاحيات المحفوظة
    if ($edit_user && $edit_user['permissions_json']) {
        $user_permissions = json_decode($edit_user['permissions_json'], true) ?: [];
    }
}
?>
<!doctype html>
<html lang="ar">
<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width,initial-scale=1">
    <title>إدارة المستخدمين</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        body{font-family:Tajawal,sans-serif;background:#f6f8fa;padding:16px;direction:rtl;}
        .card{border-radius:12px;padding:16px;margin-bottom:16px;box-shadow:0 4px 12px rgba(0,0,0,0.05);}
        .permissions-group { border: 1px solid #e0e0e0; border-radius: 8px; padding: 10px; margin-bottom: 15px; background-color: #fcfcfc; }
        .permissions-group-header { font-weight: bold; margin-bottom: 10px; padding-bottom: 5px; border-bottom: 1px dashed #ddd; }
    </style>
</head>
<body>
    <div class="container">
        <h4 class="mb-4">إدارة المستخدمين</h4>

        <?php if ($message): ?>
            <div class="alert alert-<?= $message_type ?>"><?= $message ?></div>
        <?php endif; ?>

        <div class="mb-3 d-flex justify-content-between">
            <a href="add_user.php" class="btn btn-primary">➕ إضافة مستخدم جديد</a>
            <a href="/barcode-system/admin_dashboard.php" class="btn btn-secondary">العودة للوحة التحكم</a>
        </div>

        <?php if ($edit_user): ?>
            <div class="card mb-4">
                <div class="card-header bg-warning text-dark">
                    تعديل المستخدم: **<?php echo htmlspecialchars($edit_user['username']); ?>**
                </div>
                <div class="card-body">
                    <form method="post" action="manage_users.php">
                        <input type="hidden" name="user_id_to_edit" value="<?php echo $edit_user['id']; ?>">

                        <div class="mb-3">
                            <label for="new_username" class="form-label">اسم المستخدم الجديد</label>
                            <input type="text" name="new_username" id="new_username" class="form-control" value="<?php echo htmlspecialchars($edit_user['username']); ?>" required>
                        </div>

                        <div class="mb-3">
                            <label for="new_password" class="form-label">كلمة المرور الجديدة (اختياري)</label>
                            <input type="password" name="new_password" id="new_password" class="form-control" placeholder="اترك فارغاً لعدم التغيير">
                            <small class="form-text text-muted">لتغيير كلمة المرور دون القديمة، أدخل كلمة جديدة لا تقل عن 6 أحرف.</small>
                        </div>

                        <div class="mb-3">
                            <label for="new_role" class="form-label">الدور</label>
                            <select name="new_role" id="new_role" class="form-select" required>
                                <option value="super_admin" <?php echo ($edit_user['role'] === 'super_admin' ? 'selected' : ''); ?>>مدير عام (Super Admin)</option>
                                <option value="admin" <?php echo ($edit_user['role'] === 'admin' ? 'selected' : ''); ?>>مدير مقيد (Admin)</option>
                                <option value="delivery_agent" <?php echo ($edit_user['role'] === 'delivery_agent' ? 'selected' : ''); ?>>مندوب تسليم</option>
                            </select>
                            <?php if ($edit_user['id'] == $current_user_id): ?>
                                <small class="form-text text-danger">لا يمكنك تغيير دورك من سوبر أدمن إلى دور آخر.</small>
                            <?php endif; ?>
                        </div>

                        <div class="mb-4" id="permissionsContainer">
                            <label class="form-label">الصلاحيات المخصصة (تطبق فقط إذا كان الدور **مدير مقيد** أو **مندوب تسليم**)</label>
                            <div class="row">
                                <?php foreach ($all_permissions as $group_key => $group): ?>
                                    <div class="col-md-4">
                                        <div class="permissions-group">
                                            <div class="permissions-group-header"><?= htmlspecialchars($group['label']) ?></div>
                                            <?php foreach ($group['perms'] as $perm_key => $perm_label): ?>
                                                <div class="form-check">
                                                    <input class="form-check-input"
                                                           type="checkbox"
                                                           value="<?= htmlspecialchars($perm_key) ?>"
                                                           id="<?= htmlspecialchars($perm_key) ?>"
                                                           name="permissions[]"
                                                           <?php echo in_array($perm_key, $user_permissions) ? 'checked' : ''; ?>
                                                           >
                                                    <label class="form-check-label" for="<?= htmlspecialchars($perm_key) ?>">
                                                        <?= htmlspecialchars($perm_label) ?>
                                                    </label>
                                                </div>
                                            <?php endforeach; ?>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                        </div>

                        <button type="submit" class="btn btn-success">حفظ التعديلات</button>
                        <a href="manage_users.php" class="btn btn-secondary">إلغاء</a>
                    </form>
                </div>
            </div>
        <?php endif; ?>


        <div class="card">
            <h5>قائمة المستخدمين</h5>
            <div class="table-responsive">
                <table class="table table-striped table-hover table-sm">
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>اسم المستخدم</th>
                            <th>الصلاحية</th>
                            <th>الصلاحيات المخصصة</th>
                            <th>تاريخ الإنشاء</th>
                            <th>إجراءات</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach($user_list as $u): ?>
                        <tr>
                            <td><?= $u['id'] ?></td>
                            <td>
                                <?= htmlspecialchars($u['username']) ?>
                                <?php if ($u['id'] == $current_user_id): ?>
                                    <span class="badge text-bg-info">أنت</span>
                                <?php endif; ?>
                            </td>

                            <td>
                                <?php
                                switch($u['role']) {
                                    case 'super_admin': echo '<span class="badge text-bg-danger">مدير عام</span>'; break;
                                    case 'admin': echo '<span class="badge text-bg-warning">مدير مقيد</span>'; break;
                                    case 'delivery_agent': echo '<span class="badge text-bg-success">مندوب تسليم</span>'; break;
                                    default: echo '<span class="badge text-bg-secondary">' . htmlspecialchars($u['role']) . '</span>';
                                }
                                ?>
                            </td>

                            <td class="small">
                                <?php if ($u['permissions_json'] && $u['role'] !== 'super_admin'): ?>
                                    <?php
                                        $perms_array = json_decode($u['permissions_json'], true) ?: [];
                                        echo '<span class="badge text-bg-light text-dark border">' . count($perms_array) . ' صلاحية مخصصة </span>';
                                    ?>
                                <?php elseif ($u['role'] === 'super_admin'): ?>
                                    <span class="badge text-bg-dark">الكل</span>
                                <?php else: ?>
                                    <span class="text-muted">- لا يوجد -</span>
                                <?php endif; ?>
                            </td>

                            <td><?= date('Y-m-d', strtotime($u['created_at'])) ?></td>

                            <td>
                                <a href="?edit_id=<?= $u['id'] ?>" class="btn btn-sm btn-info text-white">تعديل</a>
                                <?php if ($u['id'] !== $current_user_id): ?>
                                    <button onclick="deleteUser(<?= $u['id'] ?>,'<?= htmlspecialchars($u['username']) ?>')" class="btn btn-sm btn-danger">حذف</button>
                                <?php endif; ?>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>

    <script>
        function deleteUser(id, username) {
            if (confirm("هل أنت متأكد من حذف المستخدم '" + username + "'؟")) {
                window.location.href = `manage_users.php?action=delete&id=${id}`;
            }
        }

        // 🆕 JavaScript لإظهار/إخفاء حاوية الصلاحيات بناءً على الدور
        const roleSelect = document.getElementById('new_role');
        const permissionsContainer = document.getElementById('permissionsContainer');

        function togglePermissions() {
            // الصلاحيات المخصصة تظهر فقط للأدوار غير "super_admin"
            if (roleSelect.value === 'super_admin') {
                permissionsContainer.style.display = 'none';
            } else {
                permissionsContainer.style.display = 'block';
            }
        }

        // استدعاء عند التحميل الأولي
        if (roleSelect) {
            togglePermissions();
            // استدعاء عند تغيير قيمة القائمة المنسدلة
            roleSelect.addEventListener('change', togglePermissions);
        }
    </script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>